<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Set course image API for teachers.
 *
 * @package local_copilot
 * @author Lai Wei <lai.wei@enovation.ie>
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 * @copyright (C) 2024 onwards Microsoft, Inc. (http://microsoft.com/)
 */

namespace local_copilot\local\api_functions;

use local_copilot\manifest_generator;

/**
 * Set course image API for teachers.
 */
class local_copilot_set_course_image_for_teacher extends api_function_base {
    /**
     * Constructor.
     */
    public function __construct() {
        parent::__construct();
        $this->path = '/local_copilot_set_course_image_for_teacher';
        $this->method = 'post';
        $this->summary = 'Set course image in a course.';
        $this->description = 'This API function updates the course image in the course with the provided course ID using the ' .
            'image data provided in the request.';
        $this->operationid = 'setCourseImageForTeacher';
        $this->scopesuffix = 'write';
        $this->parameters = [];
        $this->requestbody = [
            'content' => [
                'application/json' => [
                    'schema' => [
                        'type' => 'object',
                        'required' => ['course_id', 'image_url'],
                        'properties' => [
                            'course_id' => [
                                'type' => 'integer',
                                'description' => 'ID of the course to update the image.',
                            ],
                            'image_url' => [
                                'type' => 'string',
                                'format' => 'uri',
                                'description' => 'Public available URL of an image.',
                            ],
                        ],
                    ],
                ],
            ],
        ];
        $this->responses = [
            '200' => [
                'description' => 'Whether the course image was updated successfully.',
                'content' => [
                    'application/json' => [
                        'schema' => [
                            'type' => 'object',
                            'description' => 'Whether the course image was updated successfully.',
                            'properties' => [
                                'success' => [
                                    'type' => 'boolean',
                                    'description' => 'Whether the course image was updated successfully.',
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            '400' => [
                'description' => 'Bad request.',
            ],
            '401' => [
                'description' => 'Unauthorized.',
            ],
            '403' => [
                'description' => 'The user does not have capability to update course.',
            ],
            '404' => [
                'description' => 'Course not found in Moodle, or image is not found.',
            ],
            '415' => [
                'description' => 'Unsupported media type.',
            ],
            '500' => [
                'description' => 'Internal server error.',
            ],
        ];
        $this->responsesemantics = [
            'data_path' => '$',
            'properties' => [
                'title' => '$.success',
            ],
        ];
        $this->instructions = 'You can use the setCourseImageForTeacher action to update the course image of a course. ' .
            'The "course_id" parameter accepts the ID of the course, ' .
            'and the "image_url" parameter accepts a public accessible URL of an image.' . PHP_EOL .
            'Currently the declarative agent cannot use images generated by Designer in the copilot chat directly in the ' .
            'setCourseImageForTeacher action. ' .
            'In order to set a generated image as course image, the user needs to download the image, ' .
            'upload it to a public server, and share the URL of the image to the agent.';
        $this->sortorder = 9;
        $this->enabled = false;
    }

    /**
     * Check if the API function is applicable to the given role type.
     *
     * @param string $roletype
     * @return bool
     */
    public static function check_applicable_role_type(string $roletype): bool {
        if (in_array($roletype, [manifest_generator::ROLE_TYPE_TEACHER])) {
            return true;
        }

        return false;
    }
}
