<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Equals operator (=) for SQL conditions.
 *
 * @package mod_booking
 * @copyright 2026 Wunderbyte GmbH
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace mod_booking\local\sql\operators;

/**
 * Equals operator class.
 *
 * @package mod_booking
 * @copyright 2026 Wunderbyte GmbH
 * @license http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class equals implements base_operator {
    /**
     * Get SQL snippet for equals operator.
     *
     * @param string $dbtype Database type ('postgres' or 'mysql')
     * @param string $uservalue User's profile field value
     * @param string $conditionvalue Value from the condition
     * @param string $tablealias Table alias for JSON fields
     * @param string $fieldkey JSON key for field name
     * @param string $valuekey JSON key for value
     * @return string SQL snippet
     */
    public function get_sql(
        string $dbtype,
        string $uservalue,
        string $conditionvalue,
        string $tablealias,
        string $fieldkey,
        string $valuekey
    ): string {
        if ($dbtype == 'postgres') {
            return $this->get_sql_postgres($tablealias, $fieldkey, $valuekey);
        } else {
            return $this->get_sql_mysql($tablealias, $fieldkey, $valuekey);
        }
    }

    /**
     * Get PostgreSQL specific SQL snippet for equals.
     *
     * This checks if a user profile field value needs to be retrieved from user_info_data
     * and compared with the condition value from JSON.
     *
     * @param string $objalias Object alias in PostgreSQL JSON
     * @param string $fieldkey JSON key for field name
     * @param string $valuekey JSON key for value
     * @return string SQL snippet
     */
    public function get_sql_postgres(
        string $objalias,
        string $fieldkey,
        string $valuekey
    ): string {
        global $USER;

        // We need to check if the user's profile field value equals the condition value.
        // This requires a subquery to get the user's profile field value.
        return "(
            WITH userval AS (
                SELECT uid.data AS data
                FROM {user_info_data} uid
                JOIN {user_info_field} uif ON uid.fieldid = uif.id
                WHERE uid.userid = " . (int)$USER->id . "
                AND uif.shortname = ($objalias->>'$fieldkey')::text
                LIMIT 1
            )
            SELECT (
                COALESCE((SELECT data FROM userval), '') <> ''
                AND COALESCE((SELECT data FROM userval), '') = ($objalias->>'$valuekey')::text
            )
        )";
    }

    /**
     * Get MySQL specific SQL snippet for equals.
     *
     * @param string $tablealias Table alias
     * @param string $fieldkey Field key
     * @param string $valuekey Value key
     * @return string SQL snippet
     */
    public function get_sql_mysql(
        string $tablealias,
        string $fieldkey,
        string $valuekey
    ): string {
        global $USER;

        return "(
            WITH userval AS (
                SELECT uid.data AS data
                FROM {user_info_data} uid
                JOIN {user_info_field} uif ON uid.fieldid = uif.id
                WHERE uid.userid = " . (int)$USER->id . "
                AND uif.shortname = $tablealias.$fieldkey
                LIMIT 1
            )
            SELECT (
                COALESCE((SELECT data FROM userval), '') <> ''
                AND COALESCE((SELECT data FROM userval), '') = $tablealias.$valuekey
            )
        )";
    }
}
