<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Event discovery service for dynamically discovering all Moodle events.
 *
 * @package    local_mc_plugin
 * @copyright  2025 Kerem Can Akdag
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace local_mc_plugin\local;

/**
 * Service class for discovering all available Moodle events dynamically.
 */
class event_discovery {
    /** @var string Cache key for event list */
    private const CACHE_KEY = 'event_list';

    /** @var int Cache TTL in seconds (1 hour) */
    private const CACHE_TTL = 3600;

    /**
     * Get all available Moodle events.
     *
     * Returns a cached list of all discoverable events in the Moodle installation.
     * The list is cached for performance.
     *
     * @return array Array of event information arrays
     */
    public function get_all_events(): array {
        $cache = \cache::make('local_mc_plugin', 'mc_metadata');
        $cached = $cache->get(self::CACHE_KEY);

        if ($cached !== false) {
            return $cached;
        }

        $events = $this->discover_events();
        $cache->set(self::CACHE_KEY, $events);

        return $events;
    }

    /**
     * Get events grouped by category.
     *
     * Returns events organized by component/category with alphabetical sorting.
     *
     * @return array Associative array with category names as keys and event arrays as values
     */
    public function get_events_by_category(): array {
        $events = $this->get_all_events();
        $categorized = [];

        foreach ($events as $event) {
            $category = $event['category'];
            if (!isset($categorized[$category])) {
                $categorized[$category] = [];
            }
            $categorized[$category][] = $event;
        }

        ksort($categorized);

        foreach ($categorized as $category => $categoryevents) {
            usort($categoryevents, function ($a, $b) {
                return strcmp($a['name'], $b['name']);
            });
            $categorized[$category] = $categoryevents;
        }

        return $categorized;
    }

    /**
     * Search events by query string.
     *
     * Searches event names, class names, components, and descriptions for the query.
     *
     * @param string $query Search query string
     * @return array Array of matching event information arrays
     */
    public function search_events(string $query): array {
        if (empty(trim($query))) {
            return $this->get_all_events();
        }

        $events = $this->get_all_events();
        $query = strtolower(trim($query));
        $results = [];

        foreach ($events as $event) {
            $searchable = strtolower(
                $event['name'] . ' ' .
                $event['class'] . ' ' .
                $event['component'] . ' ' .
                ($event['description'] ?? '')
            );

            if (strpos($searchable, $query) !== false) {
                $results[] = $event;
            }
        }

        return $results;
    }

    /**
     * Get detailed information about a specific event.
     *
     * @param string $eventclass Fully qualified event class name
     * @return array|null Event information array, or null if not found
     */
    public function get_event_info(string $eventclass): ?array {
        $events = $this->get_all_events();

        foreach ($events as $event) {
            if ($event['class'] === $eventclass) {
                return $event;
            }
        }

        return null;
    }

    /**
     * Clear the event cache.
     *
     * Forces a fresh discovery of events on the next request.
     *
     * @return void
     */
    public function clear_cache(): void {
        $cache = \cache::make('local_mc_plugin', 'mc_metadata');
        $cache->delete(self::CACHE_KEY);
    }

    /**
     * Discover all events from Moodle components.
     *
     * Scans all Moodle components for event classes and filters out abstract
     * and deprecated events.
     *
     * @return array Array of event information arrays
     */
    private function discover_events(): array {
        global $CFG;
        $events = [];

        $olddebug = $CFG->debug;
        $olddisplay = $CFG->debugdisplay;
        $CFG->debug = 0;
        $CFG->debugdisplay = false;

        $eventclasses = \core_component::get_component_classes_in_namespace(null, 'event');

        $CFG->debug = $olddebug;
        $CFG->debugdisplay = $olddisplay;

        foreach ($eventclasses as $eventclass => $path) {
            if (!is_subclass_of($eventclass, '\core\event\base')) {
                continue;
            }

            try {
                $reflection = new \ReflectionClass($eventclass);
                if ($reflection->isAbstract()) {
                    continue;
                }

                $doccomment = $reflection->getDocComment();
                if ($doccomment && strpos($doccomment, '@deprecated') !== false) {
                    continue;
                }
            } catch (\ReflectionException $e) {
                continue;
            }

            $events[] = [
                'class' => $eventclass,
                'name' => self::get_friendly_name($eventclass),
                'category' => $this->get_category($eventclass),
                'component' => $this->get_component($eventclass),
                'description' => $this->get_description($eventclass),
            ];
        }

        usort($events, function ($a, $b) {
            return strcmp($a['name'], $b['name']);
        });

        return $events;
    }

    /**
     * Convert event class name to friendly readable name.
     *
     * @param string $eventclass The fully qualified event class name.
     * @return string The friendly readable name.
     */
    public static function get_friendly_name(string $eventclass): string {
        $parts = explode('\\', $eventclass);
        $name = end($parts);
        $name = str_replace('_', ' ', $name);
        $name = ucwords($name);
        return $name;
    }

    /**
     * Get category name for an event based on its component.
     *
     * @param string $eventclass Fully qualified event class name
     * @return string Human-readable category name
     */
    private function get_category(string $eventclass): string {
        $component = $this->get_component($eventclass);

        if ($component === 'core') {
            return get_string('category_core', 'local_mc_plugin');
        }

        $parts = explode('_', $component, 2);
        $type = $parts[0];
        $name = isset($parts[1]) ? ucfirst($parts[1]) : '';

        switch ($type) {
            case 'mod':
                return get_string('category_activity', 'local_mc_plugin', $name);
            case 'block':
                return get_string('category_block', 'local_mc_plugin', $name);
            case 'local':
                return get_string('category_local', 'local_mc_plugin', $name);
            case 'tool':
                return get_string('category_tool', 'local_mc_plugin', $name);
            case 'report':
                return get_string('category_report', 'local_mc_plugin', $name);
            case 'enrol':
                return get_string('category_enrol', 'local_mc_plugin', $name);
            case 'auth':
                return get_string('category_auth', 'local_mc_plugin', $name);
            case 'theme':
                return get_string('category_theme', 'local_mc_plugin', $name);
            default:
                return get_string('category_other', 'local_mc_plugin', ucfirst($component));
        }
    }

    /**
     * Extract component name from event class.
     *
     * @param string $eventclass Fully qualified event class name
     * @return string Component name (e.g., "core", "mod_forum")
     */
    private function get_component(string $eventclass): string {
        $parts = explode('\\', $eventclass);

        if (count($parts) > 0) {
            $component = $parts[0];

            if ($component === 'core') {
                return 'core';
            }

            return $component;
        }

        return 'unknown';
    }

    /**
     * Get event description from the event class.
     *
     * @param string $eventclass Fully qualified event class name
     * @return string Event description (currently returns empty string)
     */
    private function get_description(string $eventclass): string {
        try {
            if (method_exists($eventclass, 'get_description')) {
                return '';
            }
        } catch (\Exception $e) {
            // Gracefully handle method access errors.
            unset($e);
        }

        return '';
    }
}
