// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Module rendering the warning box to inform the users about misleading AI results.
 *
 * @module     local_ai_manager/warningbox
 * @copyright  2024 ISB Bayern
 * @author     Philipp Memmel
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

import {getAiInfo} from 'local_ai_manager/config';
import Log from 'core/log';
import Templates from 'core/templates';
import {events} from 'local_ai_manager/events';


/**
 * Renders the warning box.
 *
 * @param {string} selectorOrElement the selector where the warning box should be rendered into
 * @param {boolean} forceMaximize whether to show the full text or if it should be collapsed in mobile view
 */
export const renderWarningBox = async(selectorOrElement, forceMaximize = false) => {
    let aiConfig = null;
    try {
        aiConfig = await getAiInfo();
    } catch (error) {
        // This typically happens if we do not have the capabilities to retrieve the AI config.
        // So we just eventually log in debug mode and do not render anything.
        Log.debug(error);
        return;
    }
    const showAiWarningLink = aiConfig.aiwarningurl.length > 0;
    const targetElement = (selectorOrElement instanceof Element) ? selectorOrElement : document.querySelector(selectorOrElement);
    const {html, js} = await Templates.renderForPromise('local_ai_manager/ai_info_warning', {
        showaiwarninglink: showAiWarningLink,
        aiwarningurl: aiConfig.aiwarningurl
    });
    Templates.appendNodeContents(targetElement, html, js);
    const textElement = targetElement.querySelector('p');
    const warningboxListener = () => {
        textElement.classList.toggle('local_ai_manager-expanded');
    };
    if (window.innerWidth <= 576 && !forceMaximize) {
        textElement.classList.add('local_ai_manager-expandable_text');
        textElement.addEventListener('click', warningboxListener);
    }
    targetElement.addEventListener(events.collapseWarningBox, () => {
        textElement.classList.add('local_ai_manager-expandable_text');
        // Remove it first in case it has already been added (in mobile view for example).
        textElement.removeEventListener('click', warningboxListener);
        textElement.addEventListener('click', warningboxListener);
    });
    targetElement.addEventListener(events.maximizeWarningBox, () => {
        textElement.classList.remove('local_ai_manager-expandable_text');
        textElement.removeEventListener('click', warningboxListener);
    });
};
