<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

namespace mod_randomactivity\external;

use external_api;
use external_description;
use external_function_parameters;
use external_multiple_structure;
use external_single_structure;
use external_value;

defined('MOODLE_INTERNAL') || die();
global $CFG;
require_once($CFG->libdir . '/externallib.php');

/**
 * AJAX webservice returning detailed information about a question and answer category by chosen answer (for choices questions).
 * @package    report_certaintyquizzes
 * @copyright  2025 Astor Bizard, 2024 Loic Delon
 * @license    https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class activity_pool_utility extends external_api {
    /**
     * Parameters for get_balanced_seed().
     * @return external_function_parameters
     */
    public static function get_balanced_seed_parameters() {
        return new external_function_parameters([
                'id' => new external_value(PARAM_INT, 'Course module ID'),
                'nactivities' => new external_value(PARAM_INT, 'Number of activities in pool'),
                'group' => new external_value(PARAM_INT, 'Group ID', VALUE_DEFAULT, 0),
        ]);
    }

    /**
     * Get seed to balance users among activities.
     * @param int $id Course module ID.
     * @param int $nactivities Number of activities in pool.
     * @param int $group Group ID.
     */
    public static function get_balanced_seed($id, $nactivities, $group) {
        global $CFG;
        [ $id, $nactivities, $group ] = array_values(self::validate_parameters(self::get_balanced_seed_parameters(), [
                'id' => $id,
                'nactivities' => $nactivities,
                'group' => $group,
        ]));
        $context = \context_module::instance($id);
        self::validate_context($context);
        require_capability('mod/randomactivity:viewactivities', $context);

        require_once($CFG->dirroot . '/mod/randomactivity/locallib.php');

        if ($nactivities == 0) {
            throw new \moodle_exception('noactivity', 'mod_randomactivity');
        }

        $activities = [];
        for ($i = 0; $i < $nactivities; $i++) {
            $activities[] = $i;
        }

        $bestseed = 0;
        $bestdist = PHP_INT_MAX;

        for ($seed = 0; $seed <= 250; $seed++) {
            $assignees = array_map('count', randomactivity_get_assignees($activities, $seed, $context, $group));
            $mean = array_sum($assignees) / ((float) $nactivities);
            $dist = 0;
            foreach ($assignees as $ncmassignees) {
                $dist += ($ncmassignees - $mean) * ($ncmassignees - $mean); // Squared distance.
            }
            if ($dist < $bestdist) {
                $bestdist = $dist;
                $bestseed = $seed;
            }
            if ($bestdist < 1) {
                break;
            }
        }

        return $bestseed;
    }

    /**
     * Return types for get_balanced_seed().
     * @return external_description
     */
    public static function get_balanced_seed_returns() {
        return new external_value(PARAM_INT, 'Best seed found to balance users');
    }

    /**
     * Parameters for get_users_assignment().
     * @return external_function_parameters
     */
    public static function get_users_assignment_parameters() {
        return new external_function_parameters([
                'id' => new external_value(PARAM_INT, 'Course module ID'),
                'activities' => new external_value(PARAM_RAW, 'Space-separated ids of activities in pool'),
                'seed' => new external_value(PARAM_RAW, 'Current seed'),
                'group' => new external_value(PARAM_INT, 'Group ID', VALUE_DEFAULT, 0),
        ]);
    }

    /**
     * Get seed to balance users among activities.
     * @param int $id Course module ID.
     * @param string $activities Space-separated ids of activities in pool.
     * @param mixed $seed Current seed.
     * @param int $group Group ID.
     */
    public static function get_users_assignment($id, $activities, $seed, $group) {
        global $CFG;
        [ $id, $activities, $seed, $group ] = array_values(self::validate_parameters(self::get_users_assignment_parameters(), [
                'id' => $id,
                'activities' => $activities,
                'seed' => $seed,
                'group' => $group,
        ]));
        $context = \context_module::instance($id);
        self::validate_context($context);
        require_capability('mod/randomactivity:manage', $context);

        require_once($CFG->dirroot . '/mod/randomactivity/locallib.php');

        return randomactivity_get_assignees_formatted($activities, $seed, $context, $group);
    }

    /**
     * Return types for get_users_assignment().
     * @return external_description
     */
    public static function get_users_assignment_returns() {
        return new external_multiple_structure(new external_value(PARAM_RAW, 'Users for activity'), 'Users for each activity');
    }
}
