// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Provides utility methods to make a html table sortable by column.
 * @copyright  Astor Bizard, 2020
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define([], function() {
    /**
     * Makes an html table sortable by creating clickable column headers and arrows.
     * @param {String} tableid The id of the table to make sortable.
     * @param {Number[]} nosortcols (optional) An array of column indexes to exclude.
     *     Negative indexes can be specified to exclude columns counting from the last.
     * @param {Number} defaultsortcol (optional) The index of the column to sort by default.
     *     This index is computed after excluding columns. Negative indexes are allowed.
     * @param {Number} nexcludedlines (optional) The number of lines to ignore at the end of the table.
     */
    function makeSortable(tableid, nosortcols = [], defaultsortcol, nexcludedlines = 0) {
        var sortdirection;
        if (defaultsortcol === null) {
            defaultsortcol = undefined;
        }
        var tableBody = document.querySelector('#' + tableid + ' tbody');
        var tableHead = document.querySelector('#' + tableid + ' thead');
        var tableHeaders = tableHead.querySelectorAll('th');
        var sortArrows = [];
        tableHeaders.forEach(function(header, i) {
            // Create sorting arrows except for excluded columns.
            if (nosortcols.indexOf(i) != -1 || nosortcols.indexOf(i - tableHeaders.length) != -1) {
                return;
            }
            var sortArrow = document.createElement('i');
            sortArrow.classList.add('icon', 'fa', 'fa-fw');
            header.append(sortArrow);
            sortArrows.push(sortArrow);
            // Setup sort arrows / headers.
            header.style.cursor = 'pointer';
            header.style.userSelect = 'none';
            header.onclick = function() {
                if (!sortArrow.dataset.active) {
                    // Change of sorting column: remove old sorting column arrow and setup the new one.
                    var previouslyActiveArrow = tableHead.querySelector('th i[data-active]');
                    if (previouslyActiveArrow) {
                        previouslyActiveArrow.classList.remove('fa-caret-down', 'fa-caret-up');
                        delete previouslyActiveArrow.dataset.active;
                    }
                    sortArrow.classList.add('fa-caret-down');
                    sortArrow.dataset.active = true;
                    sortdirection = 1;
                } else {
                    // Change sorting direction and arrow display.
                    sortdirection = -sortdirection;
                    sortArrow.classList.toggle('fa-caret-up', sortdirection != 1);
                    sortArrow.classList.toggle('fa-caret-down', sortdirection == 1);
                }
                // Sort rows.
                var rows = Array.from(tableBody.children);
                var endrows = rows.splice(-nexcludedlines, nexcludedlines);
                rows.sort(function(a, b) {
                    // Sort according to 'value' attributes or inner text.
                    var getCellValue = function(tr) {
                        var cell = tr.querySelector('.cell.c' + i);
                        return cell.querySelector('[value]')?.getAttribute('value') || cell.innerText;
                    };
                    var v1 = getCellValue(a);
                    var v2 = getCellValue(b);
                    return sortdirection * (!isNaN(v1) && !isNaN(v2) ? v1 - v2 : v1.toString().localeCompare(v2));
                })
                .forEach(function(tr, index) {
                    // Renumber first column.
                    var candidates = Array.from(tr.firstElementChild.querySelectorAll('*'));
                    candidates.push(tr.firstElementChild);
                    candidates.forEach(function(el) {
                        if (/^[0-9]+$/.test(el.innerHTML)) {
                            el.innerHTML = index + 1;
                        }
                    });
                    // Re-insert row inside table.
                    tableBody.append(tr);
                });
                // Re-insert excluded rows at the end.
                endrows.forEach(function(tr) {
                    tableBody.append(tr);
                });
            };
        });
        // Do not sort upon click on header link.
        tableHead.querySelectorAll('th a').forEach(function(link) {
            link.onclick = function(e) {
                e.stopPropagation();
            };
        });
        sortArrows[defaultsortcol].click();
    }

    return {
        makeSortable: makeSortable
    };
});