// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Provides utility methods to make a html table manually orderable.
 * @copyright  Astor Bizard, 2020
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define([], function() {
    /**
     * Make the given tables orderable, by allowing the user to drag-and-drop rows.
     * Rows need to have at least one element with class 'draghandle' to be used as a handle.
     * The table in which the drag-and-drop occurs is computed dynamically, so cloning rows from a table
     * into another one will make them orderable in the latter.
     * @param {NodeList} tables Set of tables to make orderable.
     * @param {Function} onChange A function to call every time row order is modified.
     */
    function makeOrderable(tables, onChange) {
        var initialY;
        var prevCss;
        var initialIndex;
        var currentIndex;
        var rowsHeightOffset;
        var currentTable;
        var rowBeingDragged = null;

        tables.forEach(function(table) {
            table.querySelectorAll('.draghandle').forEach(function(dragHandle) {
                dragHandle.onmousedown = function(event) {
                    event.preventDefault();
                    initialY = event.clientY;
                    currentTable = dragHandle.closest('table'); // Don't use table variable, retrieve it dynamically as it may move.
                    rowBeingDragged = dragHandle.closest('tr');
                    rowBeingDragged.classList.add('dragged');
                    prevCss = rowBeingDragged.style;
                    initialIndex = rowBeingDragged.rowIndex - 1;
                    currentIndex = initialIndex;
                    rowsHeightOffset = 0;
                };
            });
        });

        window.onmousemove = function(event) {
            if (rowBeingDragged === null) {
                return;
            }
            event.preventDefault();

            var toplimit, bottomlimit;
            var justmoved = false;
            if (currentIndex > 0) {
                var toprow = currentTable.querySelector('tbody tr:nth-child(' + currentIndex + ')');
                toplimit = toprow.getBoundingClientRect().top + (toprow.offsetHeight + rowBeingDragged.offsetHeight) / 2;
                if (event.clientY < toplimit) {
//                    toprow.insertAdjacentElement('beforestart', rowBeingDragged);
                    toprow.parentElement.insertBefore(rowBeingDragged, toprow);
                    rowsHeightOffset += toprow.offsetHeight;
                    currentIndex--;
                    justmoved = true;
                }
            }
            if (currentIndex < currentTable.querySelectorAll('tbody tr').length - 1 && !justmoved) {
                var bottomrow = currentTable.querySelector('tbody tr:nth-child(' + (currentIndex + 2) + ')');
                bottomlimit = bottomrow.getBoundingClientRect().top + (bottomrow.offsetHeight - rowBeingDragged.offsetHeight) / 2;
                if (event.clientY > bottomlimit) {
                    bottomrow.insertAdjacentElement('afterend', rowBeingDragged);
                    rowsHeightOffset -= bottomrow.offsetHeight;
                    currentIndex++;
                }
            }

            var cssOffset = event.clientY - initialY + rowsHeightOffset;
            rowBeingDragged.style.position = 'relative';
            rowBeingDragged.style.top = cssOffset + 'px';
        };
        window.onmouseup = function() {
            if (rowBeingDragged !== null) {
                rowBeingDragged.style = prevCss;
                rowBeingDragged.classList.remove('dragged');
                rowBeingDragged = null;
                if (currentIndex != initialIndex) {
                    onChange();
                }
            }
        };
    }

    return {
        makeOrderable: makeOrderable
    };
});
