// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Defines the behavior of the view page for a randomactivity.
 * @copyright  Astor Bizard, 2020
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
define([
    'jquery',
    'jqueryui',
    'core/str',
    'mod_randomactivity/orderabletable',
    'mod_randomactivity/repository',
    ], function($, jqui, Str, OrderableTable, Repository) {
    /**
     * Reads the table and updates the hidden form field accordingly.
     * Also updates the table attributes and numbering if necessary.
     * @param {Element} table The table to read and update.
     * @param {Element} activities Hidden form field containing activities ids.
     */
    function updateActivities(table, activities) {
        // In case the table is emptied, create a dummy empty row (and remove it if the table is filled).
        var tableBody = table.querySelector('tbody');
        var tablerows = tableBody.querySelectorAll('tr');
        if (tablerows.length > 1) {
            table.querySelector('.empty-row')?.remove();
        } else if (tablerows.length == 0) {
            var ncols = table.querySelectorAll('thead th').length;
            tableBody.insertAdjacentHTML('beforeend', '<tr class="empty-row lastrow"><td colspan="' + ncols + '"></td></tr>');
        }

        // Read table content and update the table.
        var newactivities = '';
        tableBody.querySelectorAll('tr:not(.empty-row)').forEach(function(tr, i) {
            tr.querySelector('td:first-child').innerHTML = i + 1;
            newactivities += tr.dataset.cmid + ' ';
            tr.classList.remove('lastrow');
        });
        tablerows[tablerows.length - 1].classList.add('lastrow');

        // Update hidden form field.
        activities.value = newactivities.trim();
    }

    /**
     * Sets up the main Random activity view form when in edit mode.
     * It allows the user to add activities to the pool, reorder them and provides seed generation facilities.
     */
    function setup() {
        var table = document.getElementById('randomactivity-activities-table');
        var form = document.getElementById('form-randomactivity-activities');
        var activities = form.querySelector('[name=activities]');
        var seed = form.querySelector('[name=seed]');
        updateActivities(table, activities); // Update in case of refresh (modifications should be lost).

        // Setup the drad-and-drop ordering of the table.
        // Also setup for the hidden table so added rows will be ready too.
        OrderableTable.makeOrderable(document.querySelectorAll('[data-role="activitytable"]'), function() {
            updateActivities(table, activities);
            form.dispatchEvent(new CustomEvent('change'));
        });

        // Buttons "Remove" for each row.
        document.querySelectorAll('[data-role="activitytable"] button[data-role="remove"]').forEach(function(button) {
            button.onclick = function() {
                var tr = button.closest('tr');
                var hiddenTableBody = document.querySelector('.activity-to-add[data-cmid="' + tr.dataset.cmid + '"] tbody');
                hiddenTableBody.append(tr);
                updateActivities(table, activities);
                form.dispatchEvent(new CustomEvent('change'));
            };
        });

        // Button "Apply changes and re-assign".
        var applyChangesButton = document.getElementById('apply-assign-button');
        applyChangesButton.setAttribute('disabled', 'disabled');

        // On form change, prompt the user before leaving the page.
        var preventUnsavedChangesUnload = function(event) {
            event.preventDefault();
            event.returnValue = '';
        };
        var changed = false;
        form.onchange = function() {
            applyChangesButton.removeAttribute('disabled');
            if (!changed) {
                changed = true;
                window.addEventListener('beforeunload', preventUnsavedChangesUnload);
            }
        };

        // Button "Cancel".
        document.getElementById('cancel-button').onclick = function() {
            // The cancel has been explicitely requested, don't prompt the user anymore.
            window.removeEventListener('beforeunload', preventUnsavedChangesUnload);
        };

        var popupDialogClasses = {
            'ui-dialog': 'popup-dialog bg-white p-3',
            'ui-dialog-content': 'my-3',
            'ui-dialog-titlebar-close': 'd-none',
            'ui-dialog-buttonpane': 'text-right',
        };

        (async function() {
            // Confirmation dialog upon submit and acitvity re-assigning.
            var $confirmReassignDialog = $('#confirm-reassign').dialog({
                autoOpen: false,
                classes: popupDialogClasses,
                title: await Str.get_string('confirm', 'moodle'),
                modal: true,
                buttons:
                [{
                    text: await Str.get_string('confirm', 'moodle'),
                    'class': 'btn btn-primary mx-1',
                    click: function() {
                        // The unloading of the page will not lose modifications, don't prompt the user anymore.
                        window.removeEventListener('beforeunload', preventUnsavedChangesUnload);
                        form.submit();
                    }
                },
                {
                    text: await Str.get_string('cancel', 'moodle'),
                    'class': 'btn btn-secondary mx-1',
                    click: function() {
                        $(this).dialog('close');
                    }
                }],
                open: function() {
                    document.querySelector(':focus').blur();
                }
            });

            applyChangesButton.onclick = function() {
                $confirmReassignDialog.dialog('open');
            };
        })();

        // Button "Seed to timestamp".
        document.getElementById('seed-timestamp').onclick = function(event) {
            event.preventDefault();
            seed.value = (new Date()).getTime();
            form.dispatchEvent(new CustomEvent('change'));
        };

        // Button "Seed to balance users".
        document.getElementById('seed-balance').onclick = async function(event) {
            event.preventDefault();

            var load = document.querySelector('.load-icon');
            load.style.display = '';
            seed.setAttribute('disabled', 'disabled');

            var id = form.querySelector('[name=id]').value;
            var group = form.querySelector('[name=group]').value;
            var nactivities = activities.value != '' ? activities.value.split(' ').length : 0;
            seed.value = await Repository.getBalancedSeed(id, nactivities, group);
            form.dispatchEvent(new CustomEvent('change'));
            seed.removeAttribute('disabled');
            load.style.display = 'none';
        };

        // Button "Refresh assignees preview".
        document.querySelector('[data-role="refresh-assignees"]').onclick = async function() {
            document.querySelector(':focus').blur();
            var id = form.querySelector('[name=id]').value;
            var group = form.querySelector('[name=group]').value;
            var assignees = await Repository.getUsersAssignment(id, activities.value, seed.value, group);
            table.querySelectorAll('tbody tr td:nth-child(3)').forEach(function(cell, i) {
               cell.innerHTML = assignees[i];
           });
        };

        // Activities addition choice filters.
        var filters = {
            selector: document.querySelector('.activity-type-selector'),
            dropdown: document.querySelector('.activity-type-dropdown'),
            selectedvalue: document.querySelector('.activity-type-selected span'),
            search: document.getElementById('activity-search'),
        };
        var activitiesList = document.querySelector('.activities-to-add-list');
        var sections = activitiesList.querySelectorAll('.list-section');

        // Apply filters and search field to activities choice.
        var applyActivitiesChoiceFilters = function() {
            var type = filters.selectedvalue.dataset.cmtype;
            var name = filters.search.value;
            var currentActivities = activities.value.split(' ');

            var noSectionToShow = true;
            sections.forEach(function(section) {
                var noActivityToAddInSection = true;
                var isExpanded = section.classList.contains('expanded');
                section.querySelectorAll('.activity-to-add').forEach(function(activity) {
                    // Activity metadata.
                    var cmid = activity.dataset.cmid;
                    var cmtype = activity.dataset.cmtype;
                    var cmname = activity.querySelector('label').innerText;

                    // Apply filters logic.
                    var isFilteredIn = (!type || cmtype == type)
                        && !currentActivities.includes(cmid + "")
                        && (name === '' || cmname.toLowerCase().includes(name.toLowerCase()));

                    // Show or hide activity according to filters.
                    activity.toggleAttribute('hidden', !isFilteredIn || !isExpanded);
                    activity.toggleAttribute('aria-hidden', !isFilteredIn || !isExpanded);

                    noActivityToAddInSection = noActivityToAddInSection && !isFilteredIn;
                });

                // Hide section if it is empty with current filters.
                section.toggleAttribute('hidden', noActivityToAddInSection);
                section.toggleAttribute('aria-hidden', noActivityToAddInSection);
                noSectionToShow = noSectionToShow && noActivityToAddInSection;
            });

            // If all sections are hidden, show a message.
            document.getElementById('no-activity-to-add-message').toggleAttribute('hidden', !noSectionToShow);
            document.getElementById('no-activity-to-add-message').toggleAttribute('aria-hidden', !noSectionToShow);
        };

        (async function() {
            // Activities addition dialog.
            var $addActivitiesDialog = $('#activities-choice').dialog({
                autoOpen: false,
                classes: popupDialogClasses,
                title: await Str.get_string('addactivities', 'mod_randomactivity'),
                modal: true,
                buttons:
                [{
                    text: await Str.get_string('addselectedactivities', 'mod_randomactivity'),
                    'class': 'btn btn-primary mx-1',
                    click: function() {
                        document.querySelectorAll('.activity-to-add').forEach(function(activity) {
                            if (activity.querySelector('input[type="checkbox"]').checked) {
                                table.querySelector('tbody').append(activity.querySelector('tr'));
                            }
                        });
                        $(this).dialog('close');
                        updateActivities(table, activities);
                        form.dispatchEvent(new CustomEvent('change'));
                    }
                },
                {
                    text: await Str.get_string('cancel', 'moodle'),
                    'class': 'btn btn-secondary mx-1',
                    click: function() {
                        $(this).dialog('close');
                    }
                }],
                open: function() {
                    document.querySelector(':focus').blur();

                    // Uncheck all checkboxes.
                    activitiesList.querySelectorAll('.activity-to-add input[type="checkbox"]').forEach(function(checkbox) {
                        checkbox.checked = false;
                    });

                    // Expand all sections.
                    sections.forEach(function(section) {
                        section.classList.add('expanded');
                    });
                    activitiesList.scrollTop = 0;

                    // Reset filters.
                    filters.selectedvalue.innerHTML = document.querySelector('.activity-type').innerHTML;
                    filters.selectedvalue.dataset.cmtype = '';
                    filters.selector.classList.remove('expanded');
                    filters.selector.classList.add('collapsed');
                    filters.selector.style.width = '';
                    filters.dropdown.style.width = '';
                    filters.search.value = '';
                    updateSearchBar();

                    // Reset activities choice.
                    applyActivitiesChoiceFilters();
                }
            });

            // Button "Add activities".
            document.getElementById('add-activities').onclick = function() {
                $addActivitiesDialog.dialog('open');
            };
        })();

        // Activity type filter dropdown menu collapse/expand.
        var toggleDropdownMenu = function() {
            filters.selector.classList.toggle('collapsed');
            filters.selector.classList.toggle('expanded');
            var newDropdownWidth;
            if (filters.selector.classList.contains('expanded')) {
                // Compute width including scrollbar.
                var dropdownwidth = filters.dropdown.offsetWidth * 2 - filters.dropdown.clientWidth;
                newDropdownWidth = Math.max(dropdownwidth, filters.selector.offsetWidth) + 'px';
            } else {
                newDropdownWidth = '';
            }
            filters.selector.style.width = newDropdownWidth;
            filters.dropdown.style.width = newDropdownWidth;
        };
        filters.selector.onclick = toggleDropdownMenu;
        filters.dropdown.onclick = toggleDropdownMenu;

        // Activity type filter dropdown menu choice.
        document.querySelectorAll('.activity-type').forEach(function(activityType) {
            activityType.onclick = function() {
                filters.selectedvalue.innerHTML = activityType.innerHTML;
                filters.selectedvalue.dataset.cmtype = activityType.dataset.cmtype;
                applyActivitiesChoiceFilters();
            };
        });

        // Activity search field.
        var updateSearchBar = function() {
            filters.search.classList.toggle('empty', filters.search.value === '');
            applyActivitiesChoiceFilters();
        };
        filters.search.onkeyup = updateSearchBar;

        document.querySelector('.search-remove').onclick = function() {
            filters.search.value = '';
            updateSearchBar();
        };

        document.querySelector('.search-icon').onclick = function() {
            filters.search.focus();
        };

        // Select / Deselect all visible activities of a section.
        activitiesList.querySelectorAll('.list-section .toggle-all button').forEach(function(toggleButton) {
            toggleButton.onclick = function(e) {
                e.stopImmediatePropagation();
                toggleButton.closest('.list-section').querySelectorAll('.activity-to-add:not([hidden]) input[type="checkbox"]')
                .forEach(function(checkbox) {
                    checkbox.checked = (toggleButton.dataset.action == 'select');
                });
            };
        });

        // Expand / Collapse a section.
        activitiesList.querySelectorAll('.section-header').forEach(function(header) {
            header.onclick = function() {
                header.closest('.list-section').classList.toggle('expanded');
                applyActivitiesChoiceFilters();
            };
        });
    }

    return {
        setup: setup
    };
});