<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Handles viewing a certificate
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once("../../config.php");
require_once("$CFG->dirroot/mod/certify/lib.php");
use mod_certify\local\credentials;
use mod_certify\event\course_module_viewed;

$id = required_param('id', PARAM_INT); // Course Module ID.

$cm = get_coursemodule_from_id('certify', $id, 0, false, MUST_EXIST);
$course = $DB->get_record('course', ['id' => $cm->course], '*', MUST_EXIST);
$certifycertificate = $DB->get_record('certify', ['id' => $cm->instance], '*', MUST_EXIST);

require_login($course->id, false, $cm);
$context = context_module::instance($cm->id);
require_capability('mod/certify:view', $context);

// Initialize $PAGE, compute blocks.
$PAGE->set_pagelayout('incourse');
$PAGE->set_url('/mod/certify/view.php', ['id' => $cm->id]);
$PAGE->set_context($context);
$PAGE->set_cm($cm);
$PAGE->set_title(format_string($certifycertificate->name));
$PAGE->set_heading(format_string($course->fullname));

$localcredentials = new credentials();

// Trigger event for module access logging.
$event = course_module_viewed::create([
    'objectid' => $certifycertificate->id,
    'context' => $context,
]);
$event->add_record_snapshot('course', $course);
$event->add_record_snapshot('certify', $certifycertificate);
$event->trigger();

echo $OUTPUT->header();

// User has admin privileges, show table of certificates.
if (has_capability('mod/certify:manage', $context)) {
    $credentialdata = get_credential($certifycertificate->credentialuuid);

    if ($credentialdata) {
        echo html_writer::tag('h2', $credentialdata->name, ['class' => 'text-center']);
        echo html_writer::start_div('container-fluid');
        echo html_writer::start_div('row justify-content-center');

        // Certificate Image
        if (!empty($credentialdata->certificate_image)) {
            $class = empty($credentialdata->badge_image) ? 'col-md-12 text-center' : 'col-md-6';
            echo html_writer::start_div($class);
            if (!empty($credentialdata->badge_image)) {
                echo html_writer::start_div(null, ['style' => 'float: right !important;']);
                echo html_writer::start_div('text-center');
            }
            echo html_writer::tag('h4', get_string('certificateheading', 'certify'), ['class' => 'mt-4 mb-2']);
            echo html_writer::empty_tag('img', [
                'src' => $credentialdata->certificate_image,
                'alt' => get_string('certificateimagealt', 'certify'),
                'class' => 'img-fluid mr-4 ml-4',
                'style' => 'max-width: 100%; border: 1px solid #ddd; margin-bottom: 10px;'
            ]);
            if (!empty($credentialdata->badge_image)) {
                echo html_writer::end_div();
                echo html_writer::end_div();
            }
            echo html_writer::end_div();
        }

        // Badge Image
        if (!empty($credentialdata->badge_image)) {
            $class = empty($credentialdata->certificate_image) ? 'col-md-12 text-center' : 'col-md-6';
            echo html_writer::start_div($class);
            if (!empty($credentialdata->certificate_image)) {
                echo html_writer::start_div(null, ['style' => 'float: left !important;']);
                echo html_writer::start_div('text-center');
            }
            echo html_writer::tag('h4', get_string('badgeheading', 'certify'), ['class' => 'mt-4 mb-2']);
            echo html_writer::empty_tag('img', [
                'src' => $credentialdata->badge_image,
                'alt' => get_string('badgeimagealt', 'certify'),
                'class' => 'img-fluid mr-4 ml-4',
                'style' => 'max-width: 200px; margin-bottom: 10px;'
            ]);
            if (!empty($credentialdata->certificate_image)) {
                echo html_writer::end_div();
                echo html_writer::end_div();
            }
            echo html_writer::end_div();
        }

        echo html_writer::end_div(); // Close row
        echo html_writer::end_div(); // Close container
    }

    // Config Button.
    $editurl = new moodle_url('/course/modedit.php', ['update' => $cm->id]);
    echo html_writer::start_div('text-center', ['style' => 'margin-top: 20px;']);
    echo html_writer::link($editurl, get_string('settings'), ['class' => 'btn btn-primary']);
    echo html_writer::end_div();
} else {
    $userscertificatelink = certify_get_recipient_sso_link($certifycertificate->credentialuuid, $USER->firstname, $USER->lastname, $USER->email);

    // Display certificate in iframe
    if ($userscertificatelink) {
        echo html_writer::start_div('certificate-iframe-container', ['style' => 'width: 100%; margin: 20px 0;']);
        $iframe_attrs = [
            'src' => $userscertificatelink,
            'style' => 'width: 100%; height: 2500px; border: 1px solid #ddd; border-radius: 4px;',
            'frameborder' => '0',
            'allowfullscreen' => 'allowfullscreen',
            'title' => get_string('certificateviewertitle', 'certify')
        ];

        echo html_writer::tag('iframe', '', $iframe_attrs);
        echo html_writer::end_div();
    } else {
        $error_msg = get_string('credentialloaderror', 'certify');
        \core\notification::error($error_msg);
    }
}

echo $OUTPUT->footer($course);