<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * This file contains the forms to create and edit an instance of this module
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die('Direct access to this script is forbidden.');

require_once($CFG->dirroot . '/course/moodleform_mod.php');
require_once($CFG->dirroot . '/mod/certify/lib.php');
require_once($CFG->dirroot . '/mod/certify/locallib.php');

use mod_certify\Html2Text\Html2Text;
use mod_certify\local\credentials;
use mod_certify\local\certificates;
use mod_certify\local\users;
use mod_certify\local\formhelper;


/**
 * Certify settings form.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class mod_certify_mod_form extends moodleform_mod
{

    /**
     * Called to define this moodle form
     *
     * @return void
     */
    public function definition()
    {
        global $DB, $COURSE, $CFG;

        $certificatesclient = new certificates();
        $usersclient = new users();

        $updatingcert = false;
        $alreadyexists = false;

        if (!extension_loaded('mbstring')) {
            throw new moodle_exception('errormbstringextension', 'certify');
        }

        if (!extension_loaded('dom')) {
            throw new moodle_exception('errordomextension', 'certify');
        }

        $dashboardurl = 'https://app.certify.one/';

        // Make sure the API key is set.
        if (!isset($CFG->certify_api_key)) {
            throw new moodle_exception('errorapikeynotset', 'certify');
        }

        // Update form init.
        if (optional_param('update', '', PARAM_INT)) {
            $updatingcert = true;
            $cmid = optional_param('update', '', PARAM_INT);
            $cm = get_coursemodule_from_id('certify', $cmid, 0, false, MUST_EXIST);
            $id = $cm->course;
            $course = $DB->get_record('course', ['id' => $id], '*', MUST_EXIST);
            $certifycertificate = $DB->get_record('certify', ['id' => $cm->instance], '*', MUST_EXIST);
        } else if (optional_param('course', '', PARAM_INT)) { // New form init.
            $id = optional_param('course', '', PARAM_INT);
            $course = $DB->get_record('course', ['id' => $id], '*', MUST_EXIST);
            // See if other certify certificates already exist for this course.
            $alreadyexists = $DB->record_exists('certify', ['course' => $id]);
        }

        // Load user data.
        $context = context_course::instance($course->id);
        $users = get_enrolled_users($context, "mod/certify:view", null, 'u.*');

        if ($updatingcert) {
            // Grab existing certificates and cross-reference emails.
            if ($certifycertificate->achievementid) {
                $userswithcredential = $usersclient->get_users_with_credentials($users, $certifycertificate->achievementid);
            } else {
                $userswithcredential = $usersclient->get_users_with_credentials($users, $certifycertificate->credentialuuid);
            }
        }

        // Load final quiz choices.
        $quizchoices = ['' => 'Select a Quiz'];
        if ($quizes = $DB->get_records_select('quiz', 'course = :course_id', ['course_id' => $id], '', 'id, name')) {
            foreach ($quizes as $quiz) {
                $quizchoices[$quiz->id] = $quiz->name;
            }
        }

        $inputstyle = ['style' => 'width: 399px'];

        // Form start.
        $mform =& $this->_form;
        $mform->addElement('hidden', 'course', $id);
        if ($updatingcert) {
            $mform->addElement('hidden', 'instance-id', $cm->instance);
        } else {
            $mform->addElement('hidden', 'instance-id', 0);
        }
        $mform->setType('instance-id', PARAM_INT);
        $mform->addElement('header', 'general', get_string('general', 'form'));

        $mform->addElement('text', 'name', get_string('activityname', 'certify'), $inputstyle);
        $mform->addRule('name', null, 'required', null, 'client');
        $mform->setType('name', PARAM_TEXT);
        $mform->setDefault('name', $course->fullname);

        if ($alreadyexists) {
            $mform->addElement('static', 'additionalactivitiesone', '', get_string('additionalactivitiesone', 'certify'));
        }

        // Load available certificates.
        $templates = ['' => 'Select a Credential'] + $certificatesclient->get_certificates();
        if (count($templates) === 1) {
            \core\notification::error(get_string('activitygroupdescription', 'certify'));
        }

        $mform->addElement('select', 'credentialuuid', get_string('certifygroup', 'certify'), $templates, $inputstyle);
        $mform->addRule('credentialuuid', null, 'required', null, 'client');
        if ($updatingcert && $certifycertificate->credentialuuid) {
            $mform->setDefault('credentialuuid', $certifycertificate->credentialuuid);
        }

        $this->standard_coursemodule_elements();
        $this->add_action_buttons();
    }

    /**
     * Called right before form submission.
     * We use it to include missing form data from mustache templates.
     *
     * @param stdClass $data passed by reference
     * @return void
     */
    public function data_postprocessing($data)
    {
        parent::data_postprocessing($data);
        $submitteddata = $this->_form->getSubmitValues();

        $formhelper = new formhelper();
        $data->coursefieldmapping = isset($submitteddata['coursefieldmapping']) ?
            $formhelper->reindexarray($submitteddata['coursefieldmapping']) : [];
        $data->coursecustomfieldmapping = isset($submitteddata['coursecustomfieldmapping']) ?
            $formhelper->reindexarray($submitteddata['coursecustomfieldmapping']) : [];
        $data->userprofilefieldmapping = isset($submitteddata['userprofilefieldmapping']) ?
            $formhelper->reindexarray($submitteddata['userprofilefieldmapping']) : [];
    }

    /**
     * Sets the default value for a mapping field in the form.
     *
     * @param MoodleQuickForm $mform The form instance to modify.
     * @param array $defaultvalues The default values for the form fields.
     * @param string $mappingname The name of the mapping field.
     * @param string $fieldname The specific field within the mapping to set.
     * @param int $num The index of the field in case of multiple fields with the same name.
     */
    private function set_mapping_field_default($mform, $defaultvalues, $mappingname, $fieldname, $num = 0)
    {
        $value = '';
        if (isset($defaultvalues[$mappingname][$num][$fieldname])) {
            $value = $defaultvalues[$mappingname][$num][$fieldname];
        }
        $mform->setDefault(
            $mappingname . '[' . $num . '][' . $fieldname . ']',
            $value
        );
    }
}
