<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Certificate module core interaction API
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/mod/quiz/locallib.php');

use mod_certify\apirest\apirest;
use mod_certify\local\credentials;
use mod_certify\local\users;
use mod_certify\local\certify;

/**
 * Checks if a user has earned a specific credential according to the activity settings
 * @param stdClass $record An Certify activity record
 * @param array $user
 * @return bool
 */
function certify_check_if_cert_earned($record, $user)
{
    global $DB;

    $earned = false;
    return $earned;
}

/**
 * Get the SSO link for a recipient
 * @param string $credentialuuid
 * @param string $firstname
 * @param string $lastname
 * @param string $email
 */
function certify_get_recipient_sso_link($credentialuuid, $firstname, $lastname, $email)
{
    global $CFG;

    $apirest = new apirest();

    $fullname = $firstname . ' ' . $lastname;

    try {
        $response = $apirest->recipient_sso_link($credentialuuid, $fullname, $email);
        return $response->url->authenticated;

    } catch (Exception $e) {
        return null;
    }
}

function get_credential($credentialuuid)
{
    global $CFG;

    $apirest = new apirest();

    try {
        $response = $apirest->credential($credentialuuid);
        return $response->data;

    } catch (Exception $e) {
        return null;
    }
}

/**
 * Serialize completion array
 *
 * @param Array $completionarray
 */
function serialize_completion_array($completionarray)
{
    return base64_encode(serialize((array) $completionarray));
}

/**
 * Unserialize completion array
 *
 * @param stdClass $completionobject
 */
function unserialize_completion_array($completionobject)
{
    return (array) unserialize(base64_decode($completionobject));
}

/**
 * Get a timestamp for when a student completed a course. This is
 * used when manually issuing certs to get a proper issue date and
 * for the course duration item. Currently checking for the date of
 * the highest quiz attempt for the final quiz specified for that
 * certify activity.
 *
 * @param stdClass $certifyrecord
 * @param stdClass $user
 */
function certify_manual_issue_completion_timestamp($certifyrecord, $user)
{
    global $DB;

    $completedtimestamp = false;

    // Set timestamp to now if no good timestamp was found.
    if ($completedtimestamp === false) {
        $completedtimestamp = time();
    }

    return (int) $completedtimestamp;
}

/**
 * Return 's' when number is bigger than 1
 *
 * @param int $number
 * @return string
 */
function certify_number_ending($number)
{
    return ($number > 1) ? 's' : '';
}

/**
 * Convert number of seconds in a string
 *
 * @param int $seconds
 * @return string
 */
function certify_seconds_to_str($seconds)
{
    $hours = floor(($seconds %= 86400) / 3600);
    if ($hours) {
        return $hours . ' hour' . certify_number_ending($hours);
    }
    $minutes = floor(($seconds %= 3600) / 60);
    if ($minutes) {
        return $minutes . ' minute' . certify_number_ending($minutes);
    }
    return $seconds . ' second' . certify_number_ending($seconds);
}
