<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Certificate module core interaction API
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

defined('MOODLE_INTERNAL') || die();
require_once($CFG->dirroot . '/mod/certify/locallib.php');

if (file_exists($CFG->dirroot . '/vendor/autoload.php')) {
    require_once($CFG->dirroot . '/vendor/autoload.php');
}

use mod_certify\local\credentials;
use mod_certify\local\users;
use mod_certify\local\certify;

/**
 * Add certificate instance.
 *
 * @param stdObject $post
 * @return array $certificate new certificate object
 */
function certify_add_instance($post)
{
    global $DB;

    $post->groupid = isset($post->groupid) ? $post->groupid : null;

    $post->instance = isset($post->instance) ? $post->instance : null;

    $certify = new certify();

    $recordid = $certify->save_record($post);

    return $recordid;
}

/**
 * Update certificate instance.
 *
 * @param stdClass $post
 * @return stdClass $certificate updated
 */
function certify_update_instance($post)
{
    // To update your certificate details, go to certify.one.
    global $DB;

    $localcredentials = new credentials();
    $usersclient = new users();
    $certify = new certify();

    // Load grade attributes for users if need to be added in the credential.
    $userids = [];
    if (isset($post->users)) {
        foreach ($post->users as $userid => $issuecertificate) {
            if ($issuecertificate) {
                $userids[] = $userid;
            }
        }
    }
    if (isset($post->unissuedusers)) {
        foreach ($post->unissuedusers as $userid => $issuecertificate) {
            if ($issuecertificate) {
                $userids[] = $userid;
            }
        }
    }

    $existingrecord = $DB->get_record('certify', ['id' => $post->instance], '*', MUST_EXIST);

    // Set completion activitied to 0 if unchecked.
    if (!property_exists($post, 'completionactivities')) {
        $post->completionactivities = 0;
    }

    return $certify->save_record($post, $existingrecord);
}

/**
 * Given an ID of an instance of this module,
 * this function will permanently delete the instance.
 *
 * @param int $id
 * @return bool true if successful
 */
function certify_delete_instance($id)
{
    global $DB;

    // Ensure the certificate exists.
    if (!$certificate = $DB->get_record('certify', ['id' => $id])) {
        return false;
    }

    return $DB->delete_records('certify', ['id' => $id]);
}

/**
 * Supported feature list
 *
 * @uses FEATURE_MOD_INTRO
 * @param string $feature FEATURE_xx constant for requested feature
 * @return mixed True if module supports feature, null if doesn't know
 */
function certify_supports($feature)
{
    switch ($feature) {
        case FEATURE_MOD_INTRO:
            return false;
        case FEATURE_BACKUP_MOODLE2:
            return true;
        default:
            return null;
    }
}
