<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_certify\local;

defined('MOODLE_INTERNAL') || die();

require_once($CFG->dirroot . '/mod/certify/locallib.php');
require_once($CFG->libdir . '/grade/grade_item.php');

use mod_certify\apirest\apirest;
use mod_certify\local\credentials;

/**
 * Local functions related to users.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class users
{
    /**
     * HTTP request apirest.
     * @var apirest
     */
    private $apirest;

    /**
     * Constructor method
     *
     * @param stdObject $apirest a mock apirest for testing.
     */
    public function __construct($apirest = null)
    {
        // A mock apirest is passed when unit testing.
        if ($apirest) {
            $this->apirest = $apirest;
        } else {
            $this->apirest = new apirest();
        }
    }

    /**
     * Receive a list of users and fetch their credentials from
     * the certify group provided.
     *
     * @param array $enrolledusers array of users
     * @param int $groupid certify group id
     * @return array the list of users
     */
    public function get_users_with_credentials($enrolledusers, $groupid = null)
    {
        $users = [];
        $certificates = [];

        if (!$enrolledusers) {
            return $users;
        }

        $certificatesmemo = [];
        if ($groupid) {
            try {
                $credentialsclient = new credentials($this->apirest);
                $certificates = $credentialsclient->get_credentials($groupid);
            } catch (\moodle_exception $e) {
                return $users;
            }

            foreach ($certificates as $certificate) {
                if (isset($certificate->url)) {
                    $credentialurl = $certificate->url;
                } else {
                    $credentialurl = 'https://certify.one/c/' . $certificate->id;
                }
                $certificatesmemo[$certificate->recipient->email] = [
                    'credentialid' => $certificate->id,
                    'credentialurl' => $credentialurl,
                ];
            }
        }

        foreach ($enrolledusers as $user) {
            if (isset($certificatesmemo[strtolower($user->email)])) {
                $certificate = $certificatesmemo[strtolower($user->email)];
            } else {
                $certificate = null;
            }

            $credentialurl = isset($certificate) ? $certificate['credentialurl'] : null;
            $credentialid = isset($certificate) ? $certificate['credentialid'] : null;
            $user = [
                'id' => $user->id,
                'email' => $user->email,
                'name' => $user->firstname . ' ' . $user->lastname,
                'credential_url' => $credentialurl,
                'credential_id' => $credentialid,
            ];
            array_push($users, $user);
        }
        return $users;
    }

    /**
     * Receive a list of users and return only those who don't have a credential
     * and they have pass the requirements for the course.
     *
     * @param array $users array of users
     * @param int $certifyinstanceid certify module id
     * @return array list of users
     */
    public function get_unissued_users($users, $certifyinstanceid = null)
    {
        global $DB;
        $unissuedusers = [];

        if ($certifyinstanceid || $certifyinstanceid != 0) {
            $certifycertificate = $DB->get_record('certify', ['id' => $certifyinstanceid], '*', MUST_EXIST);

            foreach ($users as $user) {
                if (!$user['credential_id'] && certify_check_if_cert_earned($certifycertificate, $user)) {
                    array_push($unissuedusers, $user);
                }
            }
        }

        return $unissuedusers;
    }

    /**
     * Get user grades from a grade item.
     * Returns empty array since grade attribute mapping is no longer used.
     * Kept for compatibility with event handlers.
     *
     * @param stdObject $certify the certify activity object.
     * @param array|int $userids array of user IDs or a single ID.
     * @return array empty array
     */
    public function get_user_grades($certify, $userids)
    {
        return [];
    }

    /**
     * Map user grade with custom attribute name.
     * Returns empty array since grade attribute mapping is no longer used.
     * Kept for compatibility with event handlers.
     *
     * @param stdObject $certify the certify activity object.
     * @param array $grades array of user grades.
     * @param int $userid
     * @return array empty array
     */
    public function load_user_grade_as_custom_attributes($certify, $grades, $userid)
    {
        return [];
    }
}
