<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_certify\local;

use mod_certify\apirest\apirest;

/**
 * Local functions related to credentials.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class credentials
{
    /**
     * HTTP request apirest.
     * @var apirest
     */
    private $apirest;

    /**
     * Constructor method
     *
     * @param stdObject $apirest a mock apirest for testing.
     */
    public function __construct($apirest = null)
    {
        // A mock apirest is passed when unit testing.
        if ($apirest) {
            $this->apirest = $apirest;
        } else {
            $this->apirest = new apirest();
        }
    }

    /**
     * List all of the certificates with a specific achievement id
     * @param string $credentialuuid Limit the returned Credentials to a specific credential UUID.
     * @param string|null $email Limit the returned Credentials to a specific recipient's email address.
     * @return array[stdClass] $credentials
     */
    public function get_credentials($credentialuuid, $email = null)
    {
        global $CFG;
        $pagesize = 50;
        $page = 1;

        // Maximum number of pages to request to avoid possible infinite loop.
        $looplimit = 100;
        try {
            $loop = true;
            $count = 0;
            $credentials = [];
            // Query the Certify API and loop until it returns that there is no next page.
            while ($loop === true) {
                $credentialspage = $this->apirest->get_credentials($credentialuuid, $email, $pagesize, $page);
                foreach ($credentialspage->credentials as $credential) {
                    $credentials[] = $credential;
                }

                $page++;
                $count++;
                if ($credentialspage->meta->next_page === null || $count >= $looplimit) {
                    // If the Certify API returns that there
                    // is no next page, end the loop.
                    $loop = false;
                }
            }
            return $credentials;
        } catch (\Exception $e) {
            // Throw API exception.
            // Include the achievement id that triggered the error.
            // Direct the user to certify's support.
            // Dump the achievement id to debug_info.
            $exceptionparam = new \stdClass();
            $exceptionparam->credentialuuid = $credentialuuid;
            $exceptionparam->email = $email;
            if (isset($credentialspage)) {
                $exceptionparam->last_response = $credentialspage;
            }
            throw new \moodle_exception(
                'getcredentialserror',
                'certify',
                'https://help.certify.one/hc/en-us',
                $exceptionparam
            );
        }
    }
}
