<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_certify\client;

/**
 * The curl object used to make the request.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class client
{
    /**
     * The curl object used to make the request.
     * @var curl $curl
     */
    private $curl;

    /**
     * The options object for the requests.
     * @var array $curloptions
     */
    private $curloptions;

    /**
     * The options object for the requests.
     * @var string|null $error
     */
    public $error;

    /**
     * Constructor method
     *
     * @param stdObject $curl a mock curl for testing
     */
    public function __construct($curl = null)
    {
        global $CFG;
        require_once($CFG->libdir . '/filelib.php');

        // A mock curl is passed when unit testing.
        if ($curl) {
            $this->curl = $curl;
        } else {
            $this->curl = new \curl();
        }

        $token = $CFG->certify_api_key;
        $this->curloptions = [
            'CURLOPT_RETURNTRANSFER' => true,
            'CURLOPT_FAILONERROR' => true,
            'CURLOPT_HTTPHEADER' => [
                'Authorization: Bearer ' . $token,
                'Content-Type: application/json; charset=utf-8',
                'Certify-Integration: Moodle',
            ],
        ];

        $error = null;
    }

    /**
     * Make a GET request.
     * @param string $url
     * @return stdObject
     */
    public function get($url)
    {
        // ray('client->get', $url, $this->curloptions);
        return $this->send_req($url, 'get');
    }

    /**
     * Make a POST request.
     * @param string $url
     * @param array $reqdata
     * @return stdObject
     */
    public function post($url, $reqdata)
    {
        //Use custom curl because moodle curl doesn't support multipart/form-data
        global $CFG;

        $ch = curl_init($url);

        $header = array(
            'accept: application/json',
            'authorization: Bearer ' . $CFG->certify_api_key,
            "content-type: multipart/form-data; boundary=---011000010111000001101001"
        );

        $postfields = '';

        foreach($reqdata as $key => $value) {
            $postfields .= "-----011000010111000001101001\r\n";
            $postfields .= "Content-Disposition: form-data; name=\"{$key}\"\r\n";
            $postfields .= "\r\n{$value}\r\n";
        }

        $postfields .= "-----011000010111000001101001\r\n";

        curl_setopt_array($ch, array(
            CURLOPT_ENCODING => "",
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_POSTFIELDS => $postfields,
            CURLOPT_HTTPHEADER => $header,
            CURLOPT_RETURNTRANSFER => true
        ));

        $response = curl_exec($ch);
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpcode !== 200 && $httpcode !== 201) {
            $error_data = json_decode($response);
            $error_msg = "Certify API error: " . $error_data->message;

            \core\notification::error($error_msg);
        }

        return json_decode($response)->data;
    }

    /**
     * Make a PUT request.
     * @param string $url
     * @param string $reqdata a JSON encoded string
     * @return stdObject
     */
    public function put($url, $reqdata)
    {
        // ray('client->put', $url, $this->curloptions);
        return $this->send_req($url, 'put', $reqdata);
    }

    /**
     * Call $curl method.
     * @param string $url
     * @param string $method
     * @param string $reqdata a JSON encoded string
     * @return stdObject
     */
    private function send_req($url, $method, $reqdata = null)
    {
        $curl = $this->curl;
        $response = $curl->$method($url, $reqdata, $this->curloptions);

        if ($curl->error) {
            $this->error = $curl->error;
            debugging('<div style="padding-top: 70px; font-size: 0.9rem;"><b>CERTIFY API ERROR</b> ' .
                $curl->error . '<br />' . $method . ' ' . $url . '</div>', DEBUG_DEVELOPER);
        }

        return json_decode($response);
    }
}
