<?php
// This file is part of the Certify Certificate module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

namespace mod_certify\apirest;

use mod_certify\client\client;

/**
 * Class to make requests to Certify API.
 *
 * @package    mod_certify
 * @subpackage certify
 * @copyright  Certify <dev@certify.one>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class apirest
{
    /**
     * API base URL.
     * Use `public` to make unit testing possible.
     * @var string $apiendpoint
     */
    public $apiendpoint;

    /**
     * HTTP request client.
     * @var stdObject $client
     */
    private $client;

    /**
     * Constructor method to define correct endpoints
     *
     * @param stdObject $client a mock client for testing
     */
    public function __construct($client = null)
    {
        global $CFG;

        $this->apiendpoint = 'https://api.certify.one/api/v2/';

        // A mock client is passed when unit testing.
        if ($client) {
            $this->client = $client;
        } else {
            $this->client = new client();
        }
    }

    /**
     * Get a Credential with EnvidenceItems
     * @param string $credentialuuid
     * @return stdObject
     */
    public function credential($credentialuuid)
    {
        return $this->client->get("{$this->apiendpoint}issuer-api/credential/{$credentialuuid}");
    }

    /**
     * Generaate a Single Sign On Link for a recipient for a particular credential.
     * @param string|null $credentialid
     * @param string|null $recipientname
     * @param string|null $recipientemail
     */
    public function recipient_sso_link(
        $credentialid = null,
        $recipientname = null,
        $recipientemail = null
    ) {
        if ($recipientemail) {
            $recipientemail = strtolower($recipientemail);
        }

        $data = [
            "credential_identifier" => $credentialid,
            "recipient_name" => $recipientname,
            "recipient_email" => $recipientemail
        ];

        return $this->client->post("{$this->apiendpoint}issuer-api/issued-credentials/create", $data);
    }

    /**
     * Get attribute keys
     * @param int $pagesize
     * @param int $page
     * @param string $kind - text, date, email, image
     * @return stdObject
     */
    public function search_attribute_keys($pagesize = 50, $page = 1, $kind = 'text')
    {
        $data = json_encode(['page' => $page, 'page_size' => $pagesize, 'kind' => $kind]);
        return $this->client->post("{$this->apiendpoint}attribute_keys/search", $data);
    }

    /**
     * Creates a Credential given an existing Group
     * @param string $recipientname
     * @param string $recipientemail
     * @param string $courseid
     * @param date|null $issuedon
     * @param date|null $expiredon
     * @param stdObject|null $customattributes
     * @return stdObject
     */
    public function create_credential(
        $recipientname,
        $recipientemail,
        $courseid,
        $issuedon = null,
        $expiredon = null,
        $customattributes = null
    ) {

        $data = [
            "credential" => [
                "group_id" => $courseid,
                "recipient" => [
                    "name" => $recipientname,
                    "email" => $recipientemail,
                ],
                "issued_on" => $issuedon,
                "expired_on" => $expiredon,
                "custom_attributes" => $customattributes,
            ],
        ];

        $data = json_encode($data);

        return $this->client->post("{$this->apiendpoint}credentials", $data);
    }

    /**
     * Get Credential
     * @param int $id
     * @return stdObject
     */
    public function get_credential($id)
    {
        return $this->client->get($this->apiendpoint . 'issuer-api/credentials/' . $id);
    }

    /**
     * Get all Credentials
     * @param string $pagesize
     * @param string $page
     * @return stdObject
     */
    public function get_credentials($pagesize = 50, $page = 1)
    {
        return $this->client->get("{$this->apiendpoint}issuer-api/credentials?page_size={$pagesize}&page={$page}");
    }

    /**
     * Search Credentials
     * @param int $pagesize
     * @param int $page
     * @return stdObject
     */
    public function search_credentials($pagesize = 50, $page = 1)
    {
        $data = json_encode(['page' => $page, 'page_size' => $pagesize]);
        return $this->client->post("{$this->apiendpoint}issuer-api/credentials", $data);
    }
}
