// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @module     tiny_cursive/autosaver
 * @category   TinyMCE Editor
 * @copyright  CTI <info@cursivetechnology.com>
 * @author     Brain Station 23 <sales@brainstation-23.com>
 */

import {call} from 'core/ajax';
import {create} from 'core/modal_factory';
import {get_string as getString} from 'core/str';
import {save, cancel, hidden} from 'core/modal_events';
import $ from 'jquery';
import {iconUrl, iconGrayUrl, tooltipCss} from 'tiny_cursive/common';
import Autosave from 'tiny_cursive/cursive_autosave';
import DocumentView from 'tiny_cursive/document_view';
import {call as getUser} from "core/ajax";

export const register = (editor, interval, userId, hasApiKey, MODULES, Rubrics, submission, quizInfo, pasteSetting) => {

    var isStudent = !($('#body').hasClass('teacher_admin'));
    var intervention = $('#body').hasClass('intervention');
    var host = M.cfg.wwwroot;
    var userid = userId;
    var courseid = M.cfg.courseId;
    var editorid = editor?.id;
    var cmid = M.cfg.contextInstanceId;
    var ed = "";
    var event = "";
    var filename = "";
    var questionid = 0;
    var quizSubmit = $('#mod_quiz-next-nav');
    let assignSubmit = $('#id_submitbutton');
    var syncInterval = interval ? interval * 1000 : 10000; // Default: Sync Every 10s.
    var lastCaretPos = 1;
    let aiContents = [];
    var isFullScreen = false;
    var user = null;
    let ur = window.location.href;
    let parm = new URL(ur);
    let modulesInfo = getModulesInfo(ur, parm, MODULES);
    var resourceId = modulesInfo.resourceId;
    var modulename = modulesInfo.name;
    var errorAlert = true;
    let PASTE_SETTING = pasteSetting || 'allow';
    let shouldBlockPaste = false;
    let isPasteAllowed = false;

    if (modulename !== 'assign') {
        PASTE_SETTING = 'cite_source';
    }
    const postOne = async(methodname, args) => {
        try {
            const response = await call([{
                methodname,
                args,
            }])[0];
            if (response) {
                setTimeout(() => {
                    Autosave.updateSavingState('saved');
                }, 1000);
            }
            return response;
        } catch (error) {
            Autosave.updateSavingState('offline');
            window.console.error('Error in postOne:', error);
            throw error;
        }
    };

    getUser([{
            methodname: 'core_user_get_users_by_field',
            args: {field: 'id', values: [userid]},
        }])[0].done(response => {
            user = response[0];
        }).fail((ex) => {
            window.console.error('Error fetching user data:', ex);
        });

    assignSubmit.on('click', async function(e) {
        e.preventDefault();
        if (filename) {
            // eslint-disable-next-line
            syncData().then(() => {
                assignSubmit.off('click').click();
            });
        } else {
            assignSubmit.off('click').click();
        }
        localStorage.removeItem('lastCopyCutContent');
    });

    quizSubmit.on('click', async function(e) {
        e.preventDefault();
        if (filename) {
            // eslint-disable-next-line
            syncData().then(() => {
                quizSubmit.off('click').click();
            });
        } else {
            quizSubmit.off('click').click();
        }
        localStorage.removeItem('lastCopyCutContent');
    });

    const getModal = () => {

        Promise.all([
            getString('tiny_cursive_srcurl', 'tiny_cursive'),
            getString('tiny_cursive_srcurl_des', 'tiny_cursive'),
            getString('tiny_cursive_placeholder', 'tiny_cursive')
        ]).then(function([title, titledes, placeholder]) {

            return create({
                type: 'SAVE_CANCEL',
                title: `<div><div class="tiny-cursive-title-text">${title}</div>
                <span class="tiny-cursive-title-description ">${titledes}</span></div>`,
                body: `<textarea  class="form-control inputUrl" value="" id="inputUrl" placeholder="${placeholder}"></textarea>`,
                removeOnClose: true,
            })
                .done(modal => {
                    modal.getRoot().addClass('tiny-cursive-modal');
                    modal.show();
                    var lastEvent = '';

                    modal.getRoot().on(save, function() {

                        var number = document.getElementById("inputUrl").value.trim();

                        if (number === "" || number === null || number === undefined) {
                            editor.execCommand('Undo');
                            // eslint-disable-next-line
                            getString('pastewarning', 'tiny_cursive').then(str => alert(str));
                        } else {
                            editor.execCommand('Paste');
                        }

                        postOne('cursive_user_comments', {
                            modulename: modulename,
                            cmid: cmid,
                            resourceid: resourceId,
                            courseid: courseid,
                            usercomment: number,
                            timemodified: Date.now(),
                            editorid: editorid ? editorid : ""
                        });

                        lastEvent = 'save';
                        modal.destroy();
                    });
                    modal.getRoot().on(cancel, function() {
                        editor.execCommand('Undo');
                        lastEvent = 'cancel';
                    });

                    modal.getRoot().on(hidden, function() {
                        if (lastEvent != 'cancel' && lastEvent != 'save') {
                            editor.execCommand('Undo');
                        }
                    });
                    return modal;
                });
        }).catch(error => window.console.error(error));

    };

    const sendKeyEvent = (events, editor) => {
        ed = editor;
        event = events;

        filename = `${userid}_${resourceId}_${cmid}_${modulename}_attempt`;

        if (modulename === 'quiz') {
            questionid = editorid.split(':')[1].split('_')[0];
            filename = `${userid}_${resourceId}_${cmid}_${questionid}_${modulename}_attempt`;
        }

        if (localStorage.getItem(filename)) {
            let data = JSON.parse(localStorage.getItem(filename));
            data.push({
                resourceId: resourceId,
                key: editor.key,
                keyCode: editor.keyCode,
                event: event,
                courseId: courseid,
                unixTimestamp: Date.now(),
                clientId: host,
                personId: userid,
                position: ed.caretPosition,
                rePosition: ed.rePosition,
                pastedContent: editor.pastedContent,
                aiContent: editor.aiContent
            });
            localStorage.setItem(filename, JSON.stringify(data));
        } else {
            let data = [{
                resourceId: resourceId,
                key: editor.key,
                keyCode: editor.keyCode,
                event: event,
                courseId: courseid,
                unixTimestamp: Date.now(),
                clientId: host,
                personId: userid,
                position: ed.caretPosition,
                rePosition: ed.rePosition,
                pastedContent: editor.pastedContent,
                aiContent: editor.aiContent
            }];
            localStorage.setItem(filename, JSON.stringify(data));
        }
    };

    editor.on('keyUp', (editor) => {
        customTooltip();
        let position = getCaretPosition(false);
        editor.caretPosition = position.caretPosition;
        editor.rePosition = position.rePosition;
        sendKeyEvent("keyUp", editor);
    });
    editor.on('Paste', async(e) => {
        customTooltip();
        const pastedContent = (e.clipboardData || e.originalEvent.clipboardData).getData('text');
        if (!pastedContent) {
            return;
        }
        // Trim both values for consistent comparison
        const trimmedPastedContent = pastedContent.trim();
        const lastCopyCutContent = localStorage.getItem('lastCopyCutContent');
        const isFromOwnEditor = lastCopyCutContent && trimmedPastedContent === lastCopyCutContent;

        if (isStudent && intervention) {

            if (PASTE_SETTING === 'block') {
                if (!isFromOwnEditor) {
                    e.preventDefault();
                    shouldBlockPaste = true;
                    isPasteAllowed = false;
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    getString('paste_blocked', 'tiny_cursive').then(str => {
                       return editor.windowManager.alert(str);
                    }).catch(error => window.console.error(error));
                    setTimeout(() => {
                        isPasteAllowed = true;
                        shouldBlockPaste = false;
                    }, 100);
                    return;
                }
                shouldBlockPaste = false;
                isPasteAllowed = true;
                return;
            }
            if (PASTE_SETTING === 'cite_source') {
                if (!isFromOwnEditor) {
                    e.preventDefault();
                    e.stopPropagation();
                    e.stopImmediatePropagation();
                    getModal(e);
                }
                isPasteAllowed = true;
                return;
            }
        }
        isPasteAllowed = true;
    });
    editor.on('Redo', async(e) => {
        customTooltip();
        if (isStudent && intervention) {
            getModal(e);
        }
    });
    editor.on('keyDown', (editor) => {
        customTooltip();
        const isPasteAttempt = (editor.key === 'v' || editor.key === 'V') &&
        (editor.ctrlKey || editor.metaKey);
        if (isPasteAttempt && isStudent && intervention && PASTE_SETTING === 'block' && !isPasteAllowed) {
            setTimeout(() => {
                isPasteAllowed = true;
            }, 100);
            return;
        }
        let position = getCaretPosition();
        editor.caretPosition = position.caretPosition;
        editor.rePosition = position.rePosition;
        sendKeyEvent("keyDown", editor);
    });
     editor.on('Cut', () => {
        const selectedContent = editor.selection.getContent({format: 'text'});
        localStorage.setItem('lastCopyCutContent', selectedContent.trim());
    });
    editor.on('Copy', () => {
        const selectedContent = editor.selection.getContent({format: 'text'});
        localStorage.setItem('lastCopyCutContent', selectedContent.trim());
    });
    editor.on('mouseDown', async(editor) => {
        setTimeout(() => {
            constructMouseEvent(editor);
            sendKeyEvent("mouseDown", editor);
        }, 0);
    });
    editor.on('mouseUp', async(editor) => {
        setTimeout(() => {
            constructMouseEvent(editor);
            sendKeyEvent("mouseUp", editor);
        }, 10);
    });
    editor.on('init', () => {
        customTooltip();
        localStorage.removeItem('lastCopyCutContent');
    });
    editor.on('SetContent', () => {
        customTooltip();
    });
    editor.on('FullscreenStateChanged', (e) => {
        let view = new DocumentView(user, Rubrics, submission, modulename, editor, quizInfo);
        isFullScreen = e.state;
        try {
            if (!e.state) {
                view.normalMode();
            } else {
                view.fullPageMode();
            }
        } catch (error) {
            if (errorAlert) {
                errorAlert = false;
                getString('fullmodeerror', 'tiny_cursive').then(str => {
                    return editor.windowManager.alert(str);
                }).catch(error => window.console.error(error));
            }
            view.normalMode();
            window.console.error('Error ResizeEditor event:', error);
        }
    });

    editor.on('execcommand', function(e) {
        if (e.command === "mceInsertContent") {
            const contentObj = e.value;

            const isPaste = contentObj && typeof contentObj === 'object' && contentObj.paste === true;

            let insertedContent = contentObj.content || contentObj;
            let tempDiv = document.createElement('div');
            tempDiv.innerHTML = insertedContent;
            let text = tempDiv.textContent || tempDiv.innerText || '';
            let pastedText = tempDiv.textContent || tempDiv.innerText || '';

            let position = getCaretPosition(true);
            editor.caretPosition = position.caretPosition;
            editor.rePosition = position.rePosition;

            if (isPaste) {
                if (shouldBlockPaste) {
                    shouldBlockPaste = false;
                    e.preventDefault();
                    editor.undoManager.undo();
                    return;
                }
                const lastCopyCutContent = localStorage.getItem('lastCopyCutContent');
                const isFromOwnEditor = lastCopyCutContent && pastedText.trim() === lastCopyCutContent;

                if (isStudent && intervention && PASTE_SETTING === 'block' && !isFromOwnEditor) {
                    isPasteAllowed = false;
                    editor.undoManager.undo();
                    return;
                }

                sendKeyEvent("Paste", {
                    key: "v",
                    keyCode: 86,
                    caretPosition: editor.caretPosition,
                    rePosition: editor.rePosition,
                    pastedContent: pastedText,
                    srcElement: {baseURI: window.location.href}
                });
            } else {
                aiContents.push(text);

                sendKeyEvent("aiInsert", {
                    key: "ai",
                    keyCode: 0,
                    caretPosition: editor.caretPosition,
                    rePosition: editor.rePosition,
                    aiContent: text,
                    srcElement: {baseURI: window.location.href}
                });
            }
        }
    });

    editor.on('input', function(e) {
        let position = getCaretPosition(true);
        editor.caretPosition = position.caretPosition;
        editor.rePosition = position.rePosition;
        let aiContent = e.data;

        if (e.inputType === 'insertReplacementText' || (e.inputType === 'insertText' && aiContent && aiContent.length > 1)) {

            aiContents.push(aiContent);

            e.key = "ai";
            e.keyCode = 0;
            e.caretPosition = position.caretPosition;
            e.rePosition = position.rePosition;
            e.aiContent = aiContent;

            sendKeyEvent("aiInsert", e);
        }
    });


    /**
     * Constructs a mouse event object with caret position and button information
     * @param {Object} editor - The TinyMCE editor instance
     * @function constructMouseEvent
     * @description Sets caret position, reposition, key and keyCode properties on the editor object based on current mouse state
     */
    function constructMouseEvent(editor) {
        let position = getCaretPosition(false);
        editor.caretPosition = position.caretPosition;
        editor.rePosition = position.rePosition;
        editor.key = getMouseButton(editor);
        editor.keyCode = editor.button;
    }

    /**
     * Gets the string representation of a mouse button based on its numeric value
     * @param {Object} editor - The editor object containing button information
     * @returns {string} The string representation of the mouse button ('left', 'middle', or 'right')
     */
    function getMouseButton(editor) {

        switch (editor.button) {
            case 0:
                return 'left';
            case 1:
                return 'middle';
            case 2:
                return 'right';
        }
        return null;
    }

    /**
     * Gets the current caret position in the editor
     * @param {boolean} skip - If true, returns the last known caret position instead of calculating a new one
     * @returns {Object} Object containing:
     *   - caretPosition: Sequential position number stored in session
     *   - rePosition: Absolute character offset from start of content
     * @throws {Error} Logs warning to console if error occurs during calculation
     */
    function getCaretPosition(skip = false) {
        try {
          if (!editor || !editor.selection) {
            return {caretPosition: 0, rePosition: 0};
          }

          const range = editor.selection.getRng();
          const body = editor.getBody();

          // Create a range from start of document to current caret
          const preCaretRange = range.cloneRange();
          preCaretRange.selectNodeContents(body);
          preCaretRange.setEnd(range.endContainer, range.endOffset);

          const fragment = preCaretRange.cloneContents();
          const tempDiv = document.createElement('div');
          tempDiv.appendChild(fragment);
          let textBeforeCursor = tempDiv.innerText || '';

          const endContainer = range.endContainer;
          const endOffset = range.endOffset;

          if (endOffset === 0 &&
              endContainer.nodeType === Node.ELEMENT_NODE &&
              editor.dom.isBlock(endContainer) &&
              endContainer.previousSibling) {
              textBeforeCursor += '\n';
          }
          const blockElements = tempDiv.querySelectorAll('p, div, h1, h2, h3, h4, h5, h6, li');
          let emptyBlockCount = 0;
          blockElements.forEach(block => {
            const text = block.innerText || block.textContent || '';
            if (text.trim() === '' && block.childNodes.length === 1 &&
                block.childNodes[0].nodeName === 'BR') {
              emptyBlockCount++;
            }
          });

          // Add newlines for empty blocks (these represent Enter presses that created empty lines)
          if (emptyBlockCount > 0) {
            textBeforeCursor += '\n'.repeat(emptyBlockCount);
          }

          const absolutePosition = textBeforeCursor.length;

          if (skip) {
            return {
              caretPosition: lastCaretPos,
              rePosition: absolutePosition
            };
          }
          // Increment sequential caretPosition
          const storageKey = `${userid}_${resourceId}_${cmid}_position`;
          let storedPos = parseInt(sessionStorage.getItem(storageKey), 10);
          if (isNaN(storedPos)) {
            storedPos = 0;
          }
          storedPos++;
          lastCaretPos = storedPos;
          sessionStorage.setItem(storageKey, storedPos);

          return {
            caretPosition: storedPos,
            rePosition: absolutePosition
          };

        } catch (e) {
            window.console.warn('Error getting caret position:', e);
            return {caretPosition: lastCaretPos || 1, rePosition: 0};
        }
      }


    /**
     * Synchronizes data from localStorage to server
     * @async
     * @function SyncData
     * @description Retrieves stored keypress data from localStorage and sends it to server
     * @returns {Promise} Returns response from server if data exists and is successfully sent
     * @throws {Error} Logs error to console if data submission fails
     */
    async function syncData() {

        let data = localStorage.getItem(filename);

        if (!data || data.length === 0) {
            return;
        } else {
            localStorage.removeItem(filename);
            let originalText = editor.getContent({format: 'text'});
            try {
                Autosave.updateSavingState('saving');
                // eslint-disable-next-line
                return await postOne('cursive_write_local_to_json', {
                    key: ed.key,
                    event: event,
                    keyCode: ed.keyCode,
                    resourceId: resourceId,
                    cmid: cmid,
                    modulename: modulename,
                    editorid: editorid,
                    "json_data": data,
                    originalText: originalText
                });
            } catch (error) {
                window.console.error('Error submitting data:', error);
            }
        }
    }
    /**
     * Sets up custom tooltip functionality for the Cursive icon
     * Initializes tooltip text, positions the icon in the menubar,
     * and sets up mouse event handlers for showing/hiding the tooltip
     * @function customTooltip
     */
    function customTooltip() {
        try {
            const tooltipText = getTooltipText();
            const menubarDiv = document.querySelectorAll('div[role="menubar"].tox-menubar');
            let classArray = [];

            if (menubarDiv.length) {
                menubarDiv.forEach(function(element, index) {
                    index += 1;
                    let className = 'cursive-menu-' + index;
                    element.classList.add(className);
                    classArray.push(className);
                });
            }

            const cursiveIcon = document.createElement('img');
            cursiveIcon.src = hasApiKey ? iconUrl : iconGrayUrl;

            cursiveIcon.setAttribute('class', 'tiny_cursive_StateButton');
            cursiveIcon.style.display = 'inline-block';

            cursiveState(cursiveIcon, menubarDiv, classArray);

            for (let index in classArray) {
                const elementId = "tiny_cursive_StateIcon" + index;
                const tooltipId = `tiny_cursive_tooltip${index}`;

                tooltipText.then((text) => {
                    return setTooltip(text, document.querySelector(`#${elementId}`), tooltipId);
                }).catch(error => window.console.error(error));

                $(`#${elementId}`).on('mouseenter', function() {
                    $(this).css('position', 'relative');
                    $(`#${tooltipId}`).css(tooltipCss);
                });

                $(`#${elementId}`).on('mouseleave', function() {
                    $(`#${tooltipId}`).css('display', 'none');
                });
            }
        } catch (error) {
            window.console.error('Error setting up custom tooltip:', error);
        }
    }

    /**
     * Retrieves tooltip text strings from language files
     * @async
     * @function getTooltipText
     * @returns {Promise<Object>} Object containing buttonTitle and buttonDes strings
     */
    async function getTooltipText() {
        const [
            buttonTitle,
            buttonDes,
        ] = await Promise.all([
            getString('cursive:state:active', 'tiny_cursive'),
            getString('cursive:state:active:des', 'tiny_cursive'),
        ]);
        return {buttonTitle, buttonDes};
    }

    /**
     * Updates the Cursive icon state and positions it in the menubar
     * @param {HTMLElement} cursiveIcon - The Cursive icon element to modify
     * @param {HTMLElement} menubarDiv - The menubar div element
     * @param {Array} classArray - Array of class names for the menubar div elements
     */
    function cursiveState(cursiveIcon, menubarDiv, classArray) {
        if (!menubarDiv) {
            return;
        }

        for (let index in classArray) {
            const rightWrapper = document.createElement('div');
            const imgWrapper = document.createElement('span');
            const iconClone = cursiveIcon.cloneNode(true);
            const targetMenu = document.querySelector('.' + classArray[index]);
            let elementId = "tiny_cursive_StateIcon" + index;

            rightWrapper.style.cssText = `
                        margin-left: auto;
                        display: flex;
                        align-items: center;
                    `;

            imgWrapper.id = elementId;
            imgWrapper.style.marginLeft = '.2rem';
            imgWrapper.appendChild(iconClone);
            rightWrapper.appendChild(imgWrapper);

            let moduleIds = {
                resourceId: resourceId,
                cmid: cmid,
                modulename: modulename,
                questionid: questionid,
                userid: userid,
                courseid: courseid};
            // Document mode, other modules single editor instances
            if (isFullScreen && (modulename === 'assign' || modulename === 'forum'
                || modulename === 'lesson')) {
                let existsElement = document.querySelector('.tox-menubar[class*="cursive-menu-"] > div');
                if (existsElement) {
                    existsElement.remove();
                }

                if (!document.querySelector(`#${elementId}`)) {
                    rightWrapper.style.marginTop = '3px';
                    document.querySelector('#tiny_cursive-fullpage-right-wrapper').prepend(rightWrapper);
                }

                Autosave.destroyInstance();
                Autosave.getInstance(editor, rightWrapper, moduleIds, isFullScreen);
            } else if (isFullScreen && modulename === 'quiz') { // Document mode, quiz multiple editor instances
                let existingElement = editor.container?.childNodes[1]?.childNodes[0]?.childNodes[0]?.childNodes[7];
                let newHeader = editor.container?.childNodes[0];
                if (existingElement) {
                    existingElement.remove();
                }

                if (newHeader && !newHeader.querySelector(`span[id*=tiny_cursive_StateIcon]`)) {
                    rightWrapper.style.marginTop = '3px';
                    document.querySelector('#tiny_cursive-fullpage-right-wrapper').prepend(rightWrapper);
                }
                Autosave.destroyInstance();
                Autosave.getInstance(editor, rightWrapper, moduleIds, isFullScreen);
            } else { // Regular view
                let menubar = editor?.container?.children[0]?.childNodes[0]?.childNodes[0];

                if (targetMenu && !targetMenu.querySelector(`#${elementId}`)) {
                    targetMenu.appendChild(rightWrapper);
                }
                // Regular view, multiple editor instances
                if (modulename === 'quiz' && menubar) {
                    let wrapper = menubar.querySelector('span[id*="tiny_cursive_StateIcon"]');

                    if (wrapper) {
                        Autosave.destroyInstance();
                        Autosave.getInstance(editor, wrapper?.parentElement, moduleIds, isFullScreen);
                    }
                } else {
                    Autosave.destroyInstance();
                    Autosave.getInstance(editor, rightWrapper, moduleIds, isFullScreen);
                }
            }
        }
    }

    /**
     * Sets up tooltip content and styling for the Cursive icon
     * @param {Object} text - Object containing tooltip text strings
     * @param {string} text.buttonTitle - Title text for the tooltip
     * @param {string} text.buttonDes - Description text for the tooltip
     * @param {HTMLElement} cursiveIcon - The Cursive icon element to attach tooltip to
     * @param {string} tooltipId - ID for the tooltip element
     */
    function setTooltip(text, cursiveIcon, tooltipId) {

        if (document.querySelector(`#${tooltipId}`)) {
            return;
        }
        if (cursiveIcon) {

            const tooltipSpan = document.createElement('span');
            const description = document.createElement('span');
            const linebreak = document.createElement('br');
            const tooltipTitle = document.createElement('strong');

            tooltipSpan.style.display = 'none';
            tooltipTitle.textContent = text.buttonTitle;
            tooltipTitle.style.fontSize = '16px';
            tooltipTitle.style.fontWeight = 'bold';
            description.textContent = text.buttonDes;
            description.style.fontSize = '14px';

            tooltipSpan.id = tooltipId;
            tooltipSpan.classList.add(`shadow`);
            tooltipSpan.appendChild(tooltipTitle);
            tooltipSpan.appendChild(linebreak);
            tooltipSpan.appendChild(description);
            cursiveIcon.appendChild(tooltipSpan);
        }
    }

    /**
     * Extracts module information from URL parameters
     * @param {string} ur - The base URL to analyze
     * @param {URL} parm - URL object containing search parameters
     * @param {Array} MODULES - Array of valid module names to check against
     * @returns {Object|boolean} Object containing resourceId and module name if found, false if no valid module
     */
    function getModulesInfo(ur, parm, MODULES) {
        fetchStrings();
        if (!MODULES.some(module => ur.includes(module))) {
            return false;
        }

        if (ur.includes("forum") && !ur.includes("assign")) {
            resourceId = parm.searchParams.get('edit');
        } else {
            resourceId = parm.searchParams.get('attempt');
        }

        if (resourceId === null) {
            resourceId = 0;
        }

        for (const module of MODULES) {
            if (ur.includes(module)) {
                modulename = module;
                if (module === "lesson" || module === "assign") {
                    resourceId = cmid;
                } else if (module === "oublog") {
                    resourceId = 0;
                }
                break;
            }
        }

        return {resourceId: resourceId, name: modulename};
    }

    /**
     * Fetches and caches localized strings used in the UI
     * @function fetchStrings
     * @description Retrieves strings for sidebar titles and document sidebar elements if not already cached in localStorage
     * Uses Promise.all to fetch multiple strings in parallel for better performance
     * Stores the fetched strings in localStorage under 'sbTitle' and 'docSideBar' keys
     */
    function fetchStrings() {
        if (!localStorage.getItem('sbTitle')) {
            Promise.all([
                getString('assignment', 'tiny_cursive'),
                getString('discussion', 'tiny_cursive'),
                getString('pluginname', 'mod_quiz'),
                getString('pluginname', 'mod_lesson'),
                getString('description', 'tiny_cursive'),
            ]).then(function(strings) {
                return localStorage.setItem('sbTitle', JSON.stringify(strings));
            }).catch(error => window.console.error(error));
        }
        if (!localStorage.getItem('docSideBar')) {
            Promise.all([
                getString('details', 'tiny_cursive'),
                getString('student_info', 'tiny_cursive'),
                getString('progress', 'tiny_cursive'),
                getString('description', 'tiny_cursive'),
                getString('replyingto', 'tiny_cursive'),
                getString('answeringto', 'tiny_cursive'),
                getString('importantdates', 'tiny_cursive'),
                getString('rubrics', 'tiny_cursive'),
                getString('submission_status', 'tiny_cursive'),
                getString('status', 'tiny_cursive'),
                getString('draft', 'tiny_cursive'),
                getString('draftnot', 'tiny_cursive'),
                getString('last_modified', 'tiny_cursive'),
                getString('gradings', 'tiny_cursive'),
                getString('gradenot', 'tiny_cursive'),
                getString('word_count', 'tiny_cursive'),
                getString('timeleft', 'tiny_cursive'),
                getString('nolimit', 'tiny_cursive'),
                getString('name', 'tiny_cursive'),
                getString('userename', 'tiny_cursive'),
                getString('course', 'tiny_cursive'),
                getString('opened', 'tiny_cursive'),
                getString('due', 'tiny_cursive'),
                getString('overdue', 'tiny_cursive'),
                getString('remaining', 'tiny_cursive'),
                getString('savechanges', 'tiny_cursive'),
                getString('subjectnot', 'tiny_cursive'),
                getString('remaining', 'tiny_cursive'),
            ]).then(function(strings) {
                return localStorage.setItem('docSideBar', JSON.stringify(strings));
            }).catch(error => window.console.error(error));
        }

    }

    window.addEventListener('unload', () => {
        syncData();
    });

    setInterval(syncData, syncInterval);
};
